/*==============================================================================

                                    O  F  E  L  I

                           Object  Finite  Element  Library

  ==============================================================================

   Copyright (C) 1998 - 2025 Rachid Touzani

   This file is part of OFELI.

   OFELI is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   OFELI is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with OFELI. If not, see <http://www.gnu.org/licenses/>.

  ==============================================================================

                             OFELI's Configuration File

  ==============================================================================*/

#pragma once

#include "config.h"
#include <stdexcept>
#include "util/constants.h"
#include "util/macros.h"
#include <complex>
#include <vector>

/*! \file OFELI_Config.h
 *  \ingroup Util
 *  \brief File that contains some macros.
 *
 * All these macros can be modified before compiling the library.
 */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define MY_RANDOM             55085111
typedef  unsigned long                     lsize_t;
typedef  double                            real_t;
typedef  std::vector<real_t>               Vector;
typedef  std::complex<real_t>              complex_t;
typedef  std::vector<std::complex<real_t>> CVector;
#endif /* DOXYGEN_SHOULD_SKIP_THIS */

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define XGRAPH_                0
#undef  INIT_PETSC
#endif /* DOXYGEN_SHOULD_SKIP_THIS */

/*! \def GRAPH_MEMORY
 *  \ingroup Mesh
 * \brief Memory necessary to store matrix graph.
 * \details This value is necessary only if nodes are to be renumbered.
 */
#define GRAPH_MEMORY           1000000

/*! \def MAX_NB_INPUT_FIELDS
 *  \ingroup Solver
 * \brief Maximum number of fields for an equation
 * \details Useful for coupled problems
 */
#define MAX_NB_INPUT_FIELDS                3

/*! \def MAX_NB_MESHES
 *  \ingroup Solver
 * \brief Maximum number of meshes
 * \details Useful for coupled problems
 */
#define MAX_NB_MESHES                     10

/*! \def MAX_NB_ELEMENTS
 *  \ingroup Mesh
 * \brief Maximal Number of elements
 */
#define MAX_NB_ELEMENTS                10000

/*! \def MAX_NB_NODES
 *  \ingroup Mesh
 * \brief Maximal number of nodes
 */
#define MAX_NB_NODES                   10000

/*! \def MAX_NB_SIDES
 *  \ingroup Mesh
 * \brief Maximal number of sides in.
 */
#define MAX_NB_SIDES                   30000

/*! \def MAX_NB_EDGES
 *  \ingroup Mesh
 * \brief Maximal Number of edges.
 */
#define MAX_NB_EDGES                   30000

/*! \def MAX_NBDOF_NODE
 *  \ingroup Mesh
 * \brief Maximum number of DOF supported by each node
 */
#define MAX_NBDOF_NODE                     6

/*! \def MAX_NBDOF_SIDE
 *  \ingroup Mesh
 * \brief Maximum number of DOF supported by each side
 */
#define MAX_NBDOF_SIDE                     6

/*! \def MAX_NBDOF_EDGE
 *  \ingroup Mesh
 * \brief Maximum number of DOF supported by each edge
 */
#define MAX_NBDOF_EDGE                     2

/*! \def MAX_NB_ELEMENT_NODES
 *  \ingroup Mesh
 * \brief Maximum number of nodes by element
 */
#define MAX_NB_ELEMENT_NODES              20

/*! \def MAX_NB_ELEMENT_EDGES
 *  \ingroup Mesh
 * \brief Maximum number of edges by element
 */
#define MAX_NB_ELEMENT_EDGES              10

/*! \def MAX_NB_SIDE_NODES
 *  \ingroup Mesh
 *  \brief Maximum number of nodes by side
 */
#define MAX_NB_SIDE_NODES                  9

/*! \def MAX_NB_ELEMENT_SIDES
 *  \ingroup Mesh
 *  \brief Maximum number of sides by element
 */
#define MAX_NB_ELEMENT_SIDES               8

/*! \def MAX_NB_ELEMENT_DOF
 *  \ingroup Mesh
 *  \brief Maximum number of dof by element
 */
#define MAX_NB_ELEMENT_DOF                27

/*! \def MAX_NB_SIDE_DOF
 *  \ingroup Mesh
 *  \brief Maximum number of dof by side
 */
#define MAX_NB_SIDE_DOF                    4

/*! \def MAX_NB_INT_PTS
 *  \ingroup Mesh
 *  \brief Maximum number of integration points in element
 */
#define MAX_NB_INT_PTS                    20

/*! \def MAX_NB_MATERIALS
 *  \ingroup Mesh
 *  \brief Maximum number of materials
 */
#define MAX_NB_MATERIALS                  10

/*! \def MAX_INPUT_STRING_LENGTH
 *  \ingroup IO
 *  \brief Maximum string length
 *  \details Used in class IPF
 */
#define MAX_INPUT_STRING_LENGTH          100

/*! \def FILENAME_LENGTH
 *  \ingroup IO
 *  \brief Length of a string defining a file name
 */
#define FILENAME_LENGTH                  150

/*! \def MAX_FFT_SIZE
 *  \ingroup IO
 *  \brief Maximal size for the FFT Table 
 *  This table can be used by the FFT for any number of points from 2 up to MAX_FFT_SIZE.
 *  For example, if MAX_FFT_SIZE = 14, then we can transform anywhere from 
 *  2 to 2^15 = 32,768 points, using the same sine and cosine table.
 */
#define MAX_FFT_SIZE                     15

#ifndef DOXYGEN_SHOULD_SKIP_THIS
// Macro
#define XGRAPH_          0
#define ANY              123.45678901e05 // A macro to define any number
#endif /* DOXYGEN_SHOULD_SKIP_THIS */

/*! \enum FieldName
 * \brief Enumerate variable that selects field
 */
enum FieldName {
   TEMPERATURE                  =   0,    ///< Temperature scalar field
   DISPLACEMENT                 =   1,    ///< Displacement vector field
   VELOCITY                     =   2,    ///< Velocity vector field
   PRESSURE                     =   3,    ///< Pressure scalar field
   ELECTRIC_FIELD               =   4,    ///< Electric vector field
   MAGNETIC_FIELD               =   5,    ///< Magnetic vector field
   POTENTIAL                    =   6,    ///< Potential vector field
   CONCENTRATION                =   7,    ///< Concentration field
   STRESS                       =   8,    ///< Stress vector field
   INTERNAL_ENERGY              =   9,    ///< Internal energy
   DENSITY                      =  10,    ///< Density
   MOMENTUM                     =  11,    ///< Momentum
};

/*! \enum EType
 * Enumerate variable that selects equation data type
 */
enum class EType {
      NO_TYPE            =  0,    /*!< No type prescribed                          */
      DDOMAIN            =  1,    /*!< Domain                                      */
      MESH               =  2,    /*!< Finite element mesh                         */
      GRID               =  3,    /*!< Grid                                        */
      VECTOR             =  4,    /*!< Vector or Field                             */
      MATRIX             =  5,    /*!< Matrix                                      */
      PROJECT            =  6,    /*!< Project name                                */
      INPUT              =  7,    /*!< Input                                       */
      MATERIAL           =  8,    /*!< Material data                               */
      PRESCRIBE          =  9,    /*!< Prescribe data                              */
      FUNCTION           = 10,    /*!< Function to define a tabulation             */
      INITIAL            = 11,    /*!< Initial condition or solution               */
      SOLUTION           = 11,    /*!< Initial condition or solution               */
      BOUNDARY_CONDITION = 12,    /*!< Boundary condition data                     */
      DIRICHLET          = 12,    /*!< Boundary condition data                     */
      BODY_FORCE         = 13,    /*!< Body force data                             */
      SOURCE             = 13,    /*!< Body force data                             */
      BOUNDARY_FORCE     = 14,    /*!< Body force data                             */
      FLUX               = 14,    /*!< Body force data                             */
      TRACTION           = 15,    /*!< Body force data                             */
      NEUMANN            = 15,    /*!< Body force data                             */
      POINT_FORCE        = 16,    /*!< Localized (at point) force                  */
      AUX_INPUT_FIELD_1  = 17,    /*!< Auxiliary input field 1                     */
      AUX_INPUT_FIELD_2  = 18,    /*!< Auxiliary input field 2                     */
      AUX_INPUT_FIELD_3  = 19,    /*!< Auxiliary input field 3                     */
      AUX_INPUT_FIELD_4  = 20,    /*!< Auxiliary input field 4                     */
      DISPLACEMENT       = 21,    /*!< A displacement field                        */
      VELOCITY           = 22,    /*!< A velocity field                            */
      PRESSURE           = 23,    /*!< A pressure field                            */
      TEMPERATURE        = 24     /*!< A temperature field                         */
};


/*! \enum DOFSupport
 * \brief Choose Support of degrees of freedom
 */
enum DOFSupport {
    NONE              = 0,   ///< No support assigned to DOFs
    NODE_DOF          = 1,   ///< DOFs are supported by nodes
    ELEMENT_DOF       = 2,   ///< DOFs are supported by elements
    SIDE_DOF          = 3,   ///< DOFs are supported by sides
    BOUNDARY_SIDE_DOF = 4,   ///< DOFs are supported by sides
    EDGE_DOF          = 5    ///< DOFs are supported by edges
};


/*! \enum ElementShape
 * \brief Enumerate list for element shapes
 */
enum ElementShape {
   NO_ELEMENT    = 0,            /*!< Mesh with no element              */
   POINT         = 1,            /*!< Elements are single points        */
   LINE          = 2,            /*!< Elements are segment lines        */
   TRIANGLE      = 3,            /*!< Elements are triangles            */
   QUADRILATERAL = 4,            /*!< Elements are quadrilaterals       */
   TETRAHEDRON   = 5,            /*!< Elements are tetrahedra           */
   HEXAHEDRON    = 6,            /*!< Elements are hexahedra (bricks)   */
   PENTAHEDRON   = 7,            /*!< Elements are pentahedra           */
   PRISM         = 8,            /*!< Elements are prisms               */
   PYRAMID       = 9             /*!< Elements are pyramids             */
};

/*! \brief ExternalFileFormat
 * \brief Enumerate variable that selects external file formats
 */
enum ExternalFileFormat {
   OFELI_FF,    ///< OFELI file format
   GMSH,        ///< Gmsh file format
   GNUPLOT,     ///< Gnuplot file format
   MATLAB,      ///< Matlab m-file
   VTK,         ///< VTK file format
   TECPLOT,     ///< Tecplot file format
   EASYMESH,    ///< Easymesh file format
   GAMBIT,      ///< Gambit file format
   BAMG,        ///< Bamg file format
   NETGEN,      ///< Netgen file format
   TETGEN,      ///< Tetgen file format
   TRIANGLE_FF  ///< Triangle file format
};

/*! \enum ElementType
 * \brief Choose finite element type
 */
enum ElementType {
   LINE2    =  0,            /*!< Line element with 2 nodes (P1) */
   TRIANG3  =  1,            /*!< Triangular element with 3 nodes (P1) */
   QUAD4    =  2,            /*!< Quadrilateral element with 4 nodes (Q1) */
   TETRA4   =  3,            /*!< Tetrahedral element with 4 nodes (P1) */
   HEXA8    =  4,            /*!< Hexahedral element with 8 nodes (Q1) */
   PENTA6   =  5             /*!< Pentahedral element with 6 nodes (P1*Q1) */
};

/*! \enum NonLinearIter
 * Selects iteration method for solving nonlinear problems
 */
enum NonLinearIter {
   BISECTION     =  0,    /*!< Bisection method                          */
   REGULA_FALSI  =  1,    /*!< Regula Falsi method                       */
   PICARD        =  2,    /*!< Picard's iteration method                 */
   SECANT        =  3,    /*!< Secant method                             */
   NEWTON        =  4,    /*!< Newton's method                           */
};

//#ifdef WIN32
//#define PATH_MATERIAL "c:\\Program Files\\ofeli\\material\\"
//#else
//#define PATH_MATERIAL "/usr/local/share/ofeli/material"
//#endif


#if defined(_MSC_VER) && !defined(__MWERKS__)
#define _MSVCPP_ _MSC_VER
#endif

#if defined(OFELI_COMPILED_WITH_DEBUGGING_)
#define _DEBUG
#endif

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#define OFELI_CONJ std::conj
#define OFELI_ABS  std::abs

#define NOMINMAX

#ifdef WIN32
#define PATH_SEP "\\";
#else
#define PATH_SEP "/";
#endif
#define MATERIAL_EXT      ".md"
#endif /* DOXYGEN_SHOULD_SKIP_THIS */

/**
 * \namespace OFELI
 * \brief Namespace OFELI groups all %OFELI library classes, functions, macros and global variables
 */

namespace OFELI {
/*!
 *  \addtogroup OFELI
 *  @{
 */

/**
 *  \ingroup Global
 *  \brief Verbosity parameter
 *  \details The value of Verbosity can be modified anywhere in the calling programs.
 *  It allows outputting messages in function of the used class or function.
 *  To see how this parameter is used in any class, the OFELI user has to read
 *  corresponding documentation. 
 */
    extern int Verbosity;

/**
 *  \ingroup Global
 *  \brief Time step counter
 *  \details This counter must be initialized by the user if the
 *  macro timeLoop is not used
 *  @remark May be used in conjunction with the macro TimeLoop.
 *  In this case, it has to be initialized before. Its default value is 1
 */
    extern int theStep;

/**
 *  \ingroup Global
 *  \brief Iteration counter
 *  \details This counter must be initialized by the user
 *  @remark May be used in conjunction with the macro IterationLoop.
 *  Its default value is 1
 */
    extern int theIteration;

/**
 *  \ingroup Global
 *  \brief Number of time steps
 *  @remark May be used in conjunction with the macro TimeLoop.
 */
    extern int NbTimeSteps;

/**
 *  \ingroup Global
 *  \brief Maximal number of iterations
 *  @remark May be used in conjunction with the macro IterationLoop.
 *  Its default value is 1000
 */
    extern int MaxNbIterations;

/**
 *  \ingroup Global
 *  \brief Parameter for verbosity of message outputting
 *  \details Its default value is 1
 */
    extern int Verbosity;

/**
 *  \ingroup Global
 *  \brief Time step label
 *  @remark May be used in conjunction with the macro TimeLoop.
 *  In this case, it has to be initialized before
 */
   extern real_t theTimeStep;

/**
 *  \ingroup Global
 *  \brief Time value
 *  @remark May be used in conjunction with the macro TimeLoop.
 *  Its default value is \c 0.0
 */
   extern real_t theTime;

/**
 *  \ingroup Global
 *  \brief Final time value
 *  @remark May be used in conjunction with the macro TimeLoop.
 *  In this case, it has to be initialized before
 */
   extern real_t theFinalTime;

/**
 *  \ingroup Global
 *  \brief Tolerance value for convergence
 *  @remark May be used within an iterative procedure.
 *  Its default value is \c 1.e-8
 */
   extern real_t theTolerance;

/**
 *  \ingroup Global
 *  \brief Value of discrepancy for an iterative procedure
 *  Its default value is \c 1.0
 */
   extern real_t theDiscrepancy;

/**
 *  \ingroup Global
 *  \brief Boolean variable to say if an iterative procedure has converged
 *  \details Its default value is \c false
 */
    extern bool Converged;

/**
 *  \ingroup Global
 *  Boolean to say if PETSc use was initialized.
 *  Useful only if PETSc is used
 */
    extern bool InitPetsc;

/*! @} End of Doxygen Groups */
} /* namespace OFELI */