/*==============================================================================

                                    O  F  E  L  I

                            Object  Finite  Element  Library

  ==============================================================================

   Copyright (C) 1998 - 2025 Rachid Touzani

   This file is part of OFELI.

   OFELI is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   OFELI is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with OFELI. If not, see <http://www.gnu.org/licenses/>.

  ==============================================================================

                         Implementation of class ELAS3DT4
  for 3-D Linear Elasticity Equations using 4-node tetrahedral P1 finite element

  ==============================================================================*/


#include "equations/solid/Elas3DT4.h"
#include "shape_functions/Tetra4.h"
#include "shape_functions/Triang3.h"
#include "linear_algebra/Vect_impl.h"

namespace OFELI {


Elas3DT4::Elas3DT4(Mesh& ms)
         : Equation<4,12,3,9>(ms)
{
   _equation_name = "Linearized elasticity";
   _finite_element = "3-D, 4-Node tetrahedrals (P1)";
   setMatrixType(SPARSE|SYMMETRIC);
   setSolver(CG_SOLVER,DIAG_PREC);
   _terms = int(PDE_Terms::DEVIATORIC)|int(PDE_Terms::DILATATION)|int(PDE_Terms::BODY_RHS)|
            int(PDE_Terms::BOUNDARY_RHS);
}


Elas3DT4::Elas3DT4(Mesh&         ms,
                   Vect<real_t>& u)
         : Equation<4,12,3,9>(ms,u)
{
   _equation_name = "Linearized elasticity";
   _finite_element = "3-D, 4-Node tetrahedrals (P1)";
   setMatrixType(SPARSE|SYMMETRIC);
   setSolver(CG_SOLVER,DIAG_PREC);
   _terms = int(PDE_Terms::DEVIATORIC)|int(PDE_Terms::DILATATION)|int(PDE_Terms::BODY_RHS)|
            int(PDE_Terms::BOUNDARY_RHS);
}


Elas3DT4::~Elas3DT4() { }


void Elas3DT4::set(const Element* el)
{
   _theElement = el, _theSide = nullptr;
   setMaterial();
   Tetra4 tetra(el);
   _el_geo.det = tetra.getDet();
   _el_geo.center = tetra.getCenter();
   ElementNodeCoordinates();
   if (Equa::_u!=nullptr)
      ElementNodeVector(*_u,_eu);
   if (Equa::_bf!=nullptr)
      ElementNodeVector(*_bf,_ebf);
   _dSh = tetra.DSh();
   if (_rho_set)
      _rho = _rho_fct(_el_geo.center,_TimeInt.time);
   if (_young_set)
      _young = _young_fct(_el_geo.center,_TimeInt.time);
   if (_poisson_set)
      _poisson = _poisson_fct(_el_geo.center,_TimeInt.time);
   _lambda = _poisson*_young/((1+_poisson)*(1-2*_poisson));
   _G = 0.5*_young/(1+_poisson);
   eA0 = 0, eA1 = 0, eA2 = 0;
   eMat = 0;
   eRHS = 0;
}


void Elas3DT4::set(const Side* sd)
{
   _theElement = nullptr, _theSide = sd;
   Triang3 tr(sd);
   _el_geo.area = tr.getArea();
   SideNodeCoordinates();
   if (Equa::_u!=nullptr)
      SideNodeVector(*_u,_su);
   if (Equa::_sf!=nullptr)
      SideVector(*_sf,_ssf);
   sA0 = 0;
   sRHS = 0;
}


void Elas3DT4::Media(real_t E,
                     real_t nu,
                     real_t rho)
{
   _young = E;
   _poisson = nu;
   _rho = rho;
}


void Elas3DT4::LMass(real_t coef)
{
   real_t c = 0.5*coef*OFELI_TWELVETH*_el_geo.det*_rho;
   for (size_t i=0; i<4; ++i) {
      eA2(3*i+1,3*i+1) += c;
      eA2(3*i+2,3*i+2) += c;
      eA2(3*i+3,3*i+3) += c;
   }
}


void Elas3DT4::Deviator(real_t coef)
{
   real_t c=_G*_el_geo.det*coef;
   for (size_t j=0; j<4; ++j) {
      Point<real_t> db=c*_dSh[j];
      for (size_t i=0; i<4; ++i) {
         eA0(3*i+1,3*j+1) += 2*_dSh[i].x*db.x + _dSh[i].z*db.z + _dSh[i].y*db.y;
         eA0(3*i+1,3*j+2) += _dSh[i].y*db.x;
         eA0(3*i+1,3*j+3) += _dSh[i].z*db.x;
         eA0(3*i+2,3*j+1) += _dSh[i].x*db.y;
         eA0(3*i+2,3*j+2) += 2*_dSh[i].y*db.y + _dSh[i].z*db.z + _dSh[i].x*db.x;
         eA0(3*i+2,3*j+3) += _dSh[i].z*db.y;
         eA0(3*i+3,3*j+1) += _dSh[i].x*db.z;
         eA0(3*i+3,3*j+2) += _dSh[i].y*db.z;
         eA0(3*i+3,3*j+3) += 2*_dSh[i].z*db.z + _dSh[i].y*db.y + _dSh[i].x*db.x;
      }
   }
   eMat += eA0;
}


void Elas3DT4::Dilatation(real_t coef)
{
   real_t c = _lambda*_el_geo.det*coef;
   for (size_t j=0; j<4; ++j) {
      for (size_t i=0; i<4; ++i) {
         eA0(3*i+1,3*j+1) += c*_dSh[i].x*_dSh[j].x;
         eA0(3*i+1,3*j+2) += c*_dSh[i].x*_dSh[j].y;
         eA0(3*i+1,3*j+3) += c*_dSh[i].x*_dSh[j].z;
         eA0(3*i+2,3*j+1) += c*_dSh[i].y*_dSh[j].x;
         eA0(3*i+3,3*j+2) += c*_dSh[i].y*_dSh[j].y;
         eA0(3*i+3,3*j+3) += c*_dSh[i].y*_dSh[j].z;
         eA0(3*i+3,3*j+1) += c*_dSh[i].z*_dSh[j].x;
         eA0(3*i+3,3*j+2) += c*_dSh[i].z*_dSh[j].y;
         eA0(3*i+3,3*j+3) += c*_dSh[i].z*_dSh[j].z;
      }
   }
   eMat += eA0;
}


void Elas3DT4::BodyRHS(const Vect<real_t>& f)
{
   real_t c = 0.5*OFELI_TWELVETH*_el_geo.det;
   for (size_t i=0; i<4; ++i) {
      eRHS(3*i+1) += c*f((*_theElement)(i+1)->n(),1);
      eRHS(3*i+2) += c*f((*_theElement)(i+1)->n(),2);
      eRHS(3*i+3) += c*f((*_theElement)(i+1)->n(),3);
   }
}


void Elas3DT4::BodyRHS()
{
   real_t c = _el_geo.det/24.0;
   for (size_t i=0; i<4; ++i) {
      eRHS(3*i+1) += c*_ebf(3*i  );
      eRHS(3*i+2) += c*_ebf(3*i+1);
      eRHS(3*i+3) += c*_ebf(3*i+2);
   }
}


void Elas3DT4::BoundaryRHS(const Vect<real_t>& f)
{
   for (size_t i=0; i<3; ++i) {
      if (_theSide->getCode(1) != CONTACT_BC)
         sRHS(3*i+1  ) += _el_geo.area*f(_theSide->n(),1);
      if (_theSide->getCode(2) != CONTACT_BC)
         sRHS(3*i+2) += _el_geo.area*f(_theSide->n(),2);
      if (_theSide->getCode(3) != CONTACT_BC)
         sRHS(3*i+3) += _el_geo.area*f(_theSide->n(),3);
   }
}


void Elas3DT4::BoundaryRHS()
{
   for (size_t i=0; i<3; ++i) {
      if (_theSide->getCode(1) != CONTACT_BC)
         sRHS(3*i+1) += _el_geo.area*_ssf[0];
      if (_theSide->getCode(2) != CONTACT_BC)
         sRHS(3*i+2) += _el_geo.area*_ssf[1];
      if (_theSide->getCode(3) != CONTACT_BC)
         sRHS(3*i+3) += _el_geo.area*_ssf[2];
   }
}

} /* namespace OFELI */
